% newsurv.m
% this code computes the demographic structure of the population

load sur_long;   % sur_long: contains y: the conditional survival
                 % probability table. The years for which we have the
                 % data are 1900, 1910, 1920, 1925, 1930, ..., 2060

Surtable=y;             % we have that t0=100, m=31
[t0,m]=size(Surtable);	% m*step is the span of birth years covered.
step=5;
workage=21;      % people enter the population at age workage
cutoff=10;       % and esc from the population at age t0 (=100)
                 % minus cutoff
Surtable=Surtable(workage+1:t0-cutoff, :);
T0=t0-workage-cutoff;
clear t0;

% Pop is now the growth rate of the people entering in the labor
% force at 21 years. Pop may vary over time.
% we assume that the "newborn" population increases at 1.3% yearly
% from the beginning of history (1896, so the first growth
% rate can be computed for 1897) until 1972, stays constant
% between 1972 and 1987, and decreases at .8% yearly rate from 1987
% (first negative growth in 1988) until the end of time (2060).

% the final T0 elements of Pop will later be set to a constant value
% to get a steady state (the initial setting gives pop growth from
% 1901 to 2060).

Pop=[1.013*ones(72,1); ones(15,1); .992*ones(33,1); linspace(.992,.996,40)'];

startyear=75;   
% 1975 is the year of "revelation" when people realize the problem
% associated with the SS program due to changing population.
% 1900 is the starting year of Surtable   

% Each column in Surtable is step years apart, hence we use linear interp.
% to create a new Survival table where each column is only 1 year apart.
% The matrix Survival contains the CONDITIONAL probabilities of surviving:
% the age is on the rows, the year of birth is on the columns.

clear Survival   % we may have it from previous runs

% we now create Survival: a matrix that contains the conditional prob.
% of survival for the cohorts born in each period. The first cohort we
% consider is the cohort aged T0-1 at the beginning of the period before
% the revelation: this is the first cohort that faces a non zero prob.
% of survival from the year before the revelation into the next one.
% We need the year before the revelation to get the discount factor
% in the initial steady state.
% the first cohort we consider is therefore born in startyear-1-(T0-1)
% the last one is born in 2060.

% if we need cohorts born at or before 1900, we set their conditional
% probabilities equal to the 1900 cohort and we set the growth rate
% of the newborns to the 1901 level.

if T0>=startyear,
   Survival=Surtable(:,1)*ones(1,T0+1-startyear);
   Pop=[Pop(1)*ones(T0-startyear,1); Pop];
  else
   Survival=[];
end;

% For 1901-1920 we have data every ten years. we interpolate
% linearly to get data for each year taking into account the
% frequency of the data.
Survival=[Survival Surtable(:,1)*ones(1,10)+(Surtable(:,2)- ...
        Surtable(:,1))*(1:10)/10]; % 1901-1910
Survival=[Survival Surtable(:,2)*ones(1,10)+(Surtable(:,3)- ...
        Surtable(:,2))*(1:10)/10]; % 1911-1920

% from the third column of the matrix on we have data every step years
for i=3:m-1,
  Survival=[Survival Surtable(:,i)*ones(1, step)+(Surtable(:,i+1)- ...
           Surtable(:,i))*(1:step)/step];
end,
clear m


% for all the years before the "revelation" the conditional prob.
% of survival are set equal to the probabilities corresponding to
% the CALENDAR year "year11" (this is a diagonal of Survival)
year11=70;

% to create the initial steady state the conditional prob. of surviving
% is adjusted as follows (e.g. for startyear=75, cutoff=10, year11=70):

% 1901 1902......1968 1969 1970...1975...........2060
% ------------------------------------------------------
%  x1 | x1 |     |x1  |x1 | x1 |
%  x2 | x2 |     |x2  |x2 |
%  x3 | x3 |     |x3  |
%  ...........

% in all the years before startyear the cond. prob. of surviving is
% made equal to the one of the calendar year year11. Therefore the new
% survival probability (for the considered generations) for the
% first year is given by the one of the people born in year11 (they
% are one year old in year11-1),....

% when T0>=startyear Survival has startyear-T0+1 columns before 1901;
% otherwise Survival starts in 1901.

if T0>=startyear,
   for i=1:T0-1,
     Survival(i, 1:T0-startyear+1+year11+1-i)=...
       Survival(i, T0-startyear+1+year11+1-i)*...
       ones(1,T0-startyear+1+year11+1-i);
   end;
  else,
   for i=1:T0-1,
     Survival(i, 1:year11+1-i)= Survival(i, year11+1-i)*ones(1,year11+1-i);
   end;
end;

% we discard the years that are not needed if there are any
if T0<startyear,
   Survival=Survival(:,startyear-T0:size(Survival,2));
   Pop=Pop(startyear-T0+1:length(Pop));
end;

% we need T0 extra generations to get the demographic s.s.
Survival=[Survival Survival(:,size(Survival,2))*ones(1,T0)];
Pop=[Pop; Pop(length(Pop))*ones(T0,1)];
         
[T0, n]=size(Survival);

Surabs=[ones(1,n); cumprod(Survival)];
% Surabs is the unconditional SP table, each column is a different cohort,
% each row is a different age

Demogr=zeros(T0+1, n-T0+1);
% size of each cohort at each age relative to startyear-T0 new born
% population size, with CALENDAR year in columns and age in rows.
% Demogr starts in startyear-1 and ends in 2060+T0 because the steady
% state survival probabilities start from the newborn in 2060.

% Size of new born population relative to startyear-T0. The simulation starts
% with people born in year startyear-T0
Pop1=[1; cumprod(Pop)];

% the first column corresponds to the initial steady state.
% we use the unconditional probability of survival of the initial
% steady state and its pop growth. the size of each cohort is
% Pop times the former so the older cohorts are of smaller size
% which is reflected in the appropriate use of the power for Pop.
Demogr(:,1)= Surabs(:,1).*Pop(1).^(T0-1:-1:-1)';

for i = 2:160+T0-startyear+2, % i is the calendar year
  for j= 1:T0+1, % j goes from the 21 year of age to the last one
    Demogr(j,i)=Surabs(j, i-j+T0)*Pop1(i+T0-j);
    % the jth row corresponds to people born in calendar year i-j+1
    % according to time as measured by the columns of Demogr; to
    % convert it in time measured by the columns of Surabs we take
    % into account that Surabs starts T0-1 years earlier.
  end; % for "for j"
end; % for "for i"

% total population each year relative to startyear-T0 new born
% population size.

demogr1=sum(Demogr);

% growth rate of total population.

pgr=demogr1(2:length(demogr1)) ./ demogr1(1:length(demogr1)-1);
pgr= pgr';

% percentage of a given year population at each age.

Frac = Demogr ./ (ones(T0+1,1) * demogr1);	

% percentage of retired people on population

old=[sum(Frac(45:T0+1,1:34))./sum(Frac(1:44,1:34))...
     sum(Frac(46:T0+1,35:52))./sum(Frac(1:45,35:52))...
     sum(Frac(47:T0+1,53:156))./sum(Frac(1:46,53:156))];

% plot(1974:2129, old(1:156));
% title('Dependency ratio');
% grid;
% axis([1970 2129 0.15 0.65]);

Elife=sum(Surabs)+20;
% clear Surtable y Demogr demogr1 Pop1 Surabs m i j

