clear all
set more off
set maxvar 32767

/********************************************************************
Summary: This code extracts data on causes of death from the HRS Exit Files
********************************************************************/

/********************************************************************
Load data
********************************************************************/

// Folder locations
// Modify the paths below to point to your local directory structure
local rawdata_path "$rawdata"
local output_path "$dtafiles"

*** Raw files for each wave, RAND and HRS tracker
global wave_2_exit "`rawdata_path'/randhrsexit1994v1.dta"
global wave_2a_exit "`rawdata_path'/randahdexit1995v1.dta"
global wave_3_exit "`rawdata_path'/randhrsexit1996v1.dta"
global wave_4_exit "`rawdata_path'/randhrsexit1998v1.dta"
global wave_5_exit "`rawdata_path'/randhrsexit2000v1.dta"
global wave_6_exit "`rawdata_path'/randhrsexit2002v1.dta"
global wave_7_exit "`rawdata_path'/randhrsexit2004v1.dta"
global wave_8_exit "`rawdata_path'/randhrsexit2006v1.dta"
global wave_9_exit "`rawdata_path'/randhrsexit2008v1.dta"
global wave_10_exit "`rawdata_path'/randhrsexit2010v1.dta"
global wave_11_exit "`rawdata_path'/randhrsexit2012v1.dta"
global wave_12_exit "`rawdata_path'/randhrsexit2014v1.dta"
global wave_13_exit "`rawdata_path'/randhrsexit2016v1.dta"
global wave_14_exit "`rawdata_path'/randhrsexit2018v2.dta"
global rand_hrs  "`rawdata_path'/randhrs1992_2018v2.dta"

/********************************************************************
Define Programs
********************************************************************/
capture program drop missing
program define missing
syntax varlist [if] [in], result(varname) [wave(integer 99)]

marksample touse, novarlist // process if and in statements

quietly {
    if `wave' != 99 {
        local andinwavext & inw`wave'xt == 1
    }
    else {
        local andinwave 
    }
	foreach v of varlist `varlist' {
		replace `result' = .m if (`v' == -1 | `v' == .z | `v' == 0 | `v' == .m | (`v' == . `andinwave')) & !inlist(`result',.d,.r) & (`touse') // * this is the lowest category
		}
	foreach v of varlist `varlist' {
		replace `result' = .d if (`v' == -8 | `v' == 8 | `v' == 98 | `v' == 9998 |`v' == .d | `v' == .x) & `result'!=.r & (`touse')
		}
	foreach v of varlist `varlist' {
		replace `result' = .r if (`v' == -9 | `v' == 9 | `v' == 99 | `v' == 9999 | `v' == .r | `v' == .y) & (`touse')
		}
	}
end

/********************************************************************
Load Full Set of HRS Respondents from RAND Files
********************************************************************/

use hhidpn hhid inw* pn ragender h3hhid h4hhid using "$rand_hrs"
gen in_rndhrs = 1  // Indicator that belong to the longitudinal RAND HRS Sample
					
drop if inw1 != 1 & inw2 != 1 & inw3 != 1 & inw4 != 1 & inw5 != 1 & inw6 != 1 & ///
		inw7 != 1 & inw8 != 1 & inw9 != 1 & inw10 != 1 & inw11 != 1 & ///
		inw12 != 1 & inw13 != 1 & inw14 != 1 // People that only have a 2020 interview

/********************************************************************
Define Labels
********************************************************************/

*** Wave status: Response Indicator***
label define inwxt ///
   0 "0.nonresp" ///
   1 "1.resp,deceased"
   
*** Death expected***
label define expect ///
	1 "1.expected" ///
	2 "2.unexpected" ///
	3 "3.other" ///
	.q ".q:not asked this wave"
	
*** Death Cause Grouped***
label define grpcaus ///
	1 "1.cancer" /// 
	2 "2.cardiovascular" ///
	3 "3.other" ///
	.c ".c:not a health condition" ///
	.n ".n:none" ///
	.z ".z:not yet available"
	
*** Death Cause Original***
label define causes ///
	1 "1.cancers, tumors" ///
	2 "2.skin conditions" ///
	3 "3.musculoskeletal system, connective tissue" ///
	4 "4.heart, circulatory, blood conditions" ///
	5 "5.allergies, hayfever, sinusitis, tonsillitis" ///
	6 "6.endocrine, metabolic, nutritional conditions" ///
	7 "7.digestive system (stomach, liver, gallbladder, kidney, bladder)" ///
	8 "8.neurological, sensory conditions" ///
	9 "9.reproductive system, prostate conditions" ///
	10 "10.emotional, psychological conditions" ///
	11 "11.miscellaneous" ///
	12 "12.other symptoms" ///
	13 "13.other health condition" ///
	.c ".c:not a health condition" ///
	.n ".n:none" ///
	.z ".z:not yet available"
	
*** Death Cause Simplified***
label define simplcauses ///
	1 "1.cancers, tumors" ///
	2 "2.skin conditions" ///
	3 "3.musculoskeletal system, connective tissue" ///
	4 "4.heart, circulatory, blood conditions" ///
	5 "5.allergies, hayfever, sinusitis, tonsillitis" ///
	6 "6.endocrine, metabolic, nutritional conditions" ///
	7 "7.digestive system (stomach, liver, gallbladder, kidney, bladder)" ///
	8 "8.neurological, sensory conditions" ///
	9 "9.reproductive system, prostate conditions" ///
	10 "10.emotional, psychological conditions" ///
	11 "11.miscellaneous" ///
	12 "12.other symptoms" ///
	13 "13.other health condition" ///
	14 "14.not a health condition" ///
	15 "15.missing" ///

/********************************************************************
Respondent/Deseased Information
********************************************************************/

*** Set Wave Number
local wv=2
local pre_wv=1

*** Bring Wave 2 Exit Data for Causes of Death
local coverscreen_w2_exit w116 w6867
merge 1:1 hhidpn using "$wave_2_exit", keepusing(`coverscreen_w2_exit') nogen

*** In exit interview - Wave 2
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(w116,1,.)
replace inw`wv'xt = 1 if w116==2
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
gen radexpec = .
replace radexpec = .q if inw`wv'xt==1
label variable radexpec "radexpec: r death expected"
label values radexpec expect

*** What was the major illness that led to (his/her) death?
* Grouped
gen ragcod = .
replace ragcod = .m if w6867=="" & inw`wv'xt==1
replace ragcod = .c if inlist(w6867,"E819.9","E899","E958.9","994.1")
replace ragcod = 1 if inlist(w6867,"149.0","153.9","155.2","157.9","162.9") | inlist(w6867,"174.9","183.0","185","191.9","199.1") | ///
											inlist(w6867,"202.80","208.00","208.90")
replace ragcod = 2 if inlist(w6867,"410.9","410.90","414.9","442.9","459.0") | inlist(w6867,"415.1","425.4","427.5","428.0","428.9") | ///
											inlist(w6867,"429.3","429.9","436","437.3","441.9") | w6867=="746.89"
replace ragcod = 3 if inlist(w6867,"250.00","263.9","332.0","344.81","348.1") | ///
											inlist(w6867,"486","492.8","518.89","557.0","557.9") | inlist(w6867,"569.5","571.5","577.9","710.0") | ///
											inlist(w6867,"781.9","790.2","959.9","994.1","996.62") | inlist(w6867,"997.1","998.0","E819.9","E899","E958.9") | ///
											inlist(w6867,"959.9","996.62","997.1","998")
label variable ragcod "ragcod: grouped disease that caused r's death"
label values ragcod grpcaus

*** What was the major illness that led to (his/her) death?
gen racod_h = .
replace racod_h = .m if w6867=="" & inw`wv'xt==1
replace racod_h = .c if inlist(w6867,"E819.9","E899","E958.9", "994.1")
replace racod_h = 1 if inlist(w6867,"149.0","153.9","155.2","157.9","162.9") | inlist(w6867,"174.9","183.0","185","191.9","199.1") | ///
												inlist(w6867,"202.80","208.00","208.90")
replace racod_h = 3 if w6867=="710.0"
replace racod_h = 4 if inlist(w6867,"410.9","410.90","414.9","442.9","459.0") | inlist(w6867,"415.1","425.4","427.5","428.0","428.9") | ///
											inlist(w6867,"429.3","429.9","436","437.3","441.9") | w6867=="746.89"
replace racod_h = 5 if inlist(w6867,"486","492.8","518.89")
replace racod_h = 6 if inlist(w6867,"250.00","263.9","790.2")
replace racod_h = 7 if inlist(w6867,"557.0","557.9","569.5","571.5","577.9")
replace racod_h = 8 if inlist(w6867,"332.0","344.81","348.1","781.9")
replace racod_h = 11 if inlist(w6867,"959.9","996.62","997.1","998.0") 
label variable racod_h "racod_h: disease that caused r's death"
label values racod_h causes

*** Drop Wave 2 Exit Data for Causes of Death
drop `coverscreen_w2_exit'

*********************************************************************

*** Set Wave Number
local wv=3
local pre_wv=2

*** Bring Wave 2a Exit Data for Causes of Death
local coverscreen_w2a_exit n217 n233 n234m1m n234m2m
merge 1:1 hhidpn using "$wave_2a_exit", keepusing(`coverscreen_w2a_exit') nogen

*** Bring Wave 3 Exit Data for Causes of Death
local coverscreen_w3_exit p209 p233 p234m1m p234m2m
merge 1:1 hhidpn using "$wave_3_exit", keepusing(`coverscreen_w3_exit') nogen

*** In exit interview - Wave 3
gen inw`wv'xt = .
replace inw`wv'xt = 0 if n217==. | p209==.
replace inw`wv'xt = 1 if n217==5 | p209==2
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing n233 p233 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = n233 if inrange(n233,1,2)
replace radexpec = p233 if inrange(p233,1,2)
replace radexpec = 3 if n233==7 | p233==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing n234m1m p234m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if n234m1m==998 | p234m1m==998
replace ragcod = .r if n234m1m==999 | p234m1m==999
replace ragcod = .n if p234m1m==996
replace ragcod = 1 if inrange(n234m1m,101,102) | inrange(p234m1m,101,102)
replace ragcod = 2 if inrange(n234m1m,121,129) | inrange(p234m1m,121,129)
replace ragcod = 3 if n234m1m==103 | inrange(n234m1m,111,119) | inrange(n234m1m,131,597) | n234m1m==997 | ///
											p234m1m==103 | inrange(p234m1m,111,119) | inrange(p234m1m,131,597) | p234m1m==997

*** What was the major illness that led to (his/her) death?
missing n234m1m p234m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if n234m1m==998 | p234m1m==998
replace racod_h = .r if n234m1m==999 | p234m1m==999
replace racod_h = .n if p234m1m==996
replace racod_h = 1 if inrange(n234m1m,101,102) | inrange(p234m1m,101,102)
replace racod_h = 2 if n234m1m==103 | p234m1m==103
replace racod_h = 3 if inrange(n234m1m,111,119) | inrange(p234m1m,111,119)
replace racod_h = 4 if inrange(n234m1m,121,129) | inrange(p234m1m,121,129)
replace racod_h = 5 if inrange(n234m1m,131,139) | inrange(p234m1m,131,139)
replace racod_h = 6 if inrange(n234m1m,141,149) | inrange(p234m1m,141,149)
replace racod_h = 7 if inrange(n234m1m,151,159) | inrange(p234m1m,151,159)
replace racod_h = 8 if inrange(n234m1m,161,169) | inrange(p234m1m,161,169)
replace racod_h = 9 if inrange(n234m1m,171,179) | inrange(p234m1m,171,179)
replace racod_h = 10 if inrange(n234m1m,181,189) | inrange(p234m1m,181,189)
replace racod_h = 11 if inrange(n234m1m,191,196) | inrange(p234m1m,191,196)
replace racod_h = 12 if inrange(n234m1m,595,597) | inrange(p234m1m,595,597)
replace racod_h = 13 if n234m1m==997 | p234m1m==997

*** Drop Wave 2a Exit Data for Causes of Death
drop `coverscreen_w2a_exit'

*** Drop Wave 3 Exit Data for Causes of Death
drop `coverscreen_w3_exit'

*********************************************************************

*** Set Wave Number
local wv=4
local pre_wv=3

*** Bring Wave 4 Exit Data for Causes of Death
local coverscreen_w4_exit q496 q497m1m q497m2m post_exit
merge 1:1 hhidpn using "$wave_4_exit", keepusing(`coverscreen_w4_exit') nogen

*** In exit interview - Wave 4
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(post_exit,.,1)
replace inw`wv'xt = 1 if post_exit==0
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing q496 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = q496 if inrange(q496,1,2)
replace radexpec = 3 if q496==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing q497m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if q497m1m==998
replace ragcod = .r if q497m1m==999
replace ragcod = .c if inrange(q497m1m,601,607)
replace ragcod = .n if q497m1m==996
replace ragcod = 1 if inrange(q497m1m,101,102)
replace ragcod = 2 if inrange(q497m1m,121,129)
replace ragcod = 3 if q497m1m==103 | inrange(q497m1m,111,119) | inrange(q497m1m,131,597) | q497m1m==997

*** What was the major illness that led to (his/her) death?
missing q497m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if q497m1m==998
replace racod_h = .r if q497m1m==999
replace racod_h = .c if inrange(q497m1m,601,607)
replace racod_h = .n if q497m1m==996
replace racod_h = 1 if inrange(q497m1m,101,102)
replace racod_h = 2 if q497m1m==103
replace racod_h = 3 if inrange(q497m1m,111,119)
replace racod_h = 4 if inrange(q497m1m,121,129)
replace racod_h = 5 if inrange(q497m1m,131,139)
replace racod_h = 6 if inrange(q497m1m,141,149)
replace racod_h = 7 if inrange(q497m1m,151,159)
replace racod_h = 8 if inrange(q497m1m,161,169)
replace racod_h = 9 if inrange(q497m1m,171,179)
replace racod_h = 10 if inrange(q497m1m,181,189)
replace racod_h = 11 if inrange(q497m1m,191,196)
replace racod_h = 12 if inrange(q497m1m,595,597)
replace racod_h = 13 if q497m1m==997

*** Drop Wave 4 Exit Data for Causes of Death
drop `coverscreen_w4_exit'

*********************************************************************

*** Set Wave Number
local wv=5
local pre_wv=4

*** Bring Wave 5 Exit Data for Causes of Death
local coverscreen_w5_exit r530 r531m1m r531m2m post_exit
merge 1:1 hhidpn using "$wave_5_exit", keepusing(`coverscreen_w5_exit') nogen


*** In exit interview - Wave 5
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(post_exit,.,1)
replace inw`wv'xt = 1 if post_exit==0
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing r530 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = r530 if inrange(r530,1,2)
replace radexpec = 3 if r530==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing r531m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if r531m1m==998
replace ragcod = .r if r531m1m==999
replace ragcod = .c if inrange(r531m1m,601,607)
replace ragcod = .n if r531m1m==996
replace ragcod = 1 if inrange(r531m1m,101,102)
replace ragcod = 2 if inrange(r531m1m,121,129)
replace ragcod = 3 if r531m1m==103 | inrange(r531m1m,111,119) | inrange(r531m1m,131,597) | r531m1m==997

*** What was the major illness that led to (his/her) death?
missing r531m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if r531m1m==998
replace racod_h = .r if r531m1m==999
replace racod_h = .c if inrange(r531m1m,601,607)
replace racod_h = .n if r531m1m==996
replace racod_h = 1 if inrange(r531m1m,101,102)
replace racod_h = 2 if r531m1m==103
replace racod_h = 3 if inrange(r531m1m,111,119)
replace racod_h = 4 if inrange(r531m1m,121,129)
replace racod_h = 5 if inrange(r531m1m,131,139)
replace racod_h = 6 if inrange(r531m1m,141,149)
replace racod_h = 7 if inrange(r531m1m,151,159)
replace racod_h = 8 if inrange(r531m1m,161,169)
replace racod_h = 9 if inrange(r531m1m,171,179)
replace racod_h = 10 if inrange(r531m1m,181,189)
replace racod_h = 11 if inrange(r531m1m,191,196)
replace racod_h = 12 if inrange(r531m1m,595,597)
replace racod_h = 13 if r531m1m==997

*** Drop Wave 5 Exit Data for Causes of Death
drop `coverscreen_w5_exit'

*********************************************************************

*** Set Wave Number
local wv=6
local pre_wv=5

*** Bring Wave 6 Exit Data for Causes of Death
local coverscreen_w6_exit sa028 sa131 sa133m1m sa133m2m sz145
merge 1:1 hhidpn using "$wave_6_exit", keepusing(`coverscreen_w6_exit') nogen

*** In exit interview - Wave 6
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(sz145,.,2,3)
replace inw`wv'xt = 1 if sz145==. & inrange(sa028,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

drop if inlist(hhidpn,11863010,203802010) & inw6xt==1

*** Was the death expected at about the time it occurred, or was it unexpected?
missing sa131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = sa131 if inrange(sa131,1,2)
replace radexpec = 3 if sa131==97

*** What was the major illness that led to (his/her) death?
* Grouped
missing sa133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if sa133m1m==998
replace ragcod = .r if sa133m1m==999
replace ragcod = .c if inrange(sa133m1m,601,607)
replace ragcod = .n if sa133m1m==996
replace ragcod = 1 if inrange(sa133m1m,101,102)
replace ragcod = 2 if inrange(sa133m1m,121,129)
replace ragcod = 3 if sa133m1m==103 | inrange(sa133m1m,111,119) | inrange(sa133m1m,131,597) | sa133m1m==997

*** What was the major illness that led to (his/her) death?
missing sa133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if sa133m1m==998
replace racod_h = .r if sa133m1m==999
replace racod_h = .c if inrange(sa133m1m,601,607)
replace racod_h = .n if sa133m1m==996
replace racod_h = 1 if inrange(sa133m1m,101,102)
replace racod_h = 2 if sa133m1m==103
replace racod_h = 3 if inrange(sa133m1m,111,119)
replace racod_h = 4 if inrange(sa133m1m,121,129)
replace racod_h = 5 if inrange(sa133m1m,131,139)
replace racod_h = 6 if inrange(sa133m1m,141,149)
replace racod_h = 7 if inrange(sa133m1m,151,159)
replace racod_h = 8 if inrange(sa133m1m,161,169)
replace racod_h = 9 if inrange(sa133m1m,171,179)
replace racod_h = 10 if inrange(sa133m1m,181,189)
replace racod_h = 11 if inrange(sa133m1m,191,196)
replace racod_h = 12 if inrange(sa133m1m,595,597)
replace racod_h = 13 if sa133m1m==997

*** Drop Wave 6 Exit Data for Causes of Death
drop `coverscreen_w6_exit'

*********************************************************************

*** Set Wave Number
local wv=7
local pre_wv=6

*** Bring Wave 6 Exit Data for Causes of Death
local coverscreen_w7_exit ta131 ta133m1m ta133m2m ta167 tz145 
merge 1:1 hhidpn using "$wave_7_exit", keepusing(`coverscreen_w7_exit') nogen

*** In exit interview - Wave 7
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(tz145,.,2,3,4)
replace inw`wv'xt = 1 if tz145==. & inrange(ta167,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing ta131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = ta131 if inrange(ta131,1,2)
replace radexpec = 3 if ta131==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing ta133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if ta133m1m==998
replace ragcod = .r if ta133m1m==999
replace ragcod = .c if inrange(ta133m1m,601,607)
replace ragcod = .n if ta133m1m==996
replace ragcod = 1 if inrange(ta133m1m,101,102)
replace ragcod = 2 if inrange(ta133m1m,121,129)
replace ragcod = 3 if ta133m1m==103 | inrange(ta133m1m,111,119) | inrange(ta133m1m,131,597) | ta133m1m==997

*** What was the major illness that led to (his/her) death?
missing ta133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if ta133m1m==998
replace racod_h = .r if ta133m1m==999
replace racod_h = .c if inrange(ta133m1m,601,607)
replace racod_h = .n if ta133m1m==996
replace racod_h = 1 if inrange(ta133m1m,101,102)
replace racod_h = 2 if ta133m1m==103
replace racod_h = 3 if inrange(ta133m1m,111,119)
replace racod_h = 4 if inrange(ta133m1m,121,129)
replace racod_h = 5 if inrange(ta133m1m,131,139)
replace racod_h = 6 if inrange(ta133m1m,141,149)
replace racod_h = 7 if inrange(ta133m1m,151,159)
replace racod_h = 8 if inrange(ta133m1m,161,169)
replace racod_h = 9 if inrange(ta133m1m,171,179)
replace racod_h = 10 if inrange(ta133m1m,181,189)
replace racod_h = 11 if inrange(ta133m1m,191,196)
replace racod_h = 12 if inrange(ta133m1m,595,597)
replace racod_h = 13 if ta133m1m==997

*** Drop Wave 7 Exit Data for Causes of Death
drop `coverscreen_w7_exit'

*********************************************************************

*** Set Wave Number
local wv=8
local pre_wv=7

*** Bring Wave 8 Exit Data for Causes of Death
local coverscreen_w8_exit ua131 ua133m1m ua133m2m ua167 uz145 
merge 1:1 hhidpn using "$wave_8_exit", keepusing(`coverscreen_w8_exit') nogen

*** In exit interview - Wave 8
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(uz145,.,2,3)
replace inw`wv'xt = 1 if uz145==. & inrange(ua167,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing ua131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = ua131 if inrange(ua131,1,2)
replace radexpec = 3 if ua131==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing ua133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if ua133m1m==998
replace ragcod = .r if ua133m1m==999
replace ragcod = .c if inrange(ua133m1m,601,607)
replace ragcod = .n if ua133m1m==996
replace ragcod = 1 if inrange(ua133m1m,101,102)
replace ragcod = 2 if inrange(ua133m1m,121,129)
replace ragcod = 3 if ua133m1m==103 | inrange(ua133m1m,111,119) | inrange(ua133m1m,131,597) | ua133m1m==997

*** What was the major illness that led to (his/her) death?
missing ua133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if ua133m1m==998
replace racod_h = .r if ua133m1m==999
replace racod_h = .c if inrange(ua133m1m,601,607)
replace racod_h = .n if ua133m1m==996
replace racod_h = 1 if inrange(ua133m1m,101,102)
replace racod_h = 2 if ua133m1m==103
replace racod_h = 3 if inrange(ua133m1m,111,119)
replace racod_h = 4 if inrange(ua133m1m,121,129)
replace racod_h = 5 if inrange(ua133m1m,131,139)
replace racod_h = 6 if inrange(ua133m1m,141,149)
replace racod_h = 7 if inrange(ua133m1m,151,159)
replace racod_h = 8 if inrange(ua133m1m,161,169)
replace racod_h = 9 if inrange(ua133m1m,171,179)
replace racod_h = 10 if inrange(ua133m1m,181,189)
replace racod_h = 11 if inrange(ua133m1m,191,196)
replace racod_h = 12 if inrange(ua133m1m,595,597)
replace racod_h = 13 if ua133m1m==997

*** Drop Wave 8 Exit Data for Causes of Death
drop `coverscreen_w8_exit'

*********************************************************************

*** Set Wave Number
local wv=9
local pre_wv=8

*** Bring Wave 9 Exit Data for Causes of Death
local coverscreen_w9_exit va131 va133m1m va133m2m va167 vz145 
merge 1:1 hhidpn using "$wave_9_exit", keepusing(`coverscreen_w9_exit') nogen

*** In exit interview - Wave 9
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(vz145,.,2,3)
replace inw`wv'xt = 1 if vz145==. & inrange(va167,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing va131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = va131 if inrange(va131,1,2)
replace radexpec = 3 if va131==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing va133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if va133m1m==998
replace ragcod = .r if va133m1m==999
replace ragcod = .c if inrange(va133m1m,601,607)
replace ragcod = .n if va133m1m==996
replace ragcod = 1 if inrange(va133m1m,101,102)
replace ragcod = 2 if inrange(va133m1m,121,129)
replace ragcod = 3 if va133m1m==103 | inrange(va133m1m,111,119) | inrange(va133m1m,131,597) | va133m1m==997

*** What was the major illness that led to (his/her) death?
missing va133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if va133m1m==998
replace racod_h = .r if va133m1m==999
replace racod_h = .c if inrange(va133m1m,601,607)
replace racod_h = .n if va133m1m==996
replace racod_h = 1 if inrange(va133m1m,101,102)
replace racod_h = 2 if va133m1m==103
replace racod_h = 3 if inrange(va133m1m,111,119)
replace racod_h = 4 if inrange(va133m1m,121,129)
replace racod_h = 5 if inrange(va133m1m,131,139)
replace racod_h = 6 if inrange(va133m1m,141,149)
replace racod_h = 7 if inrange(va133m1m,151,159)
replace racod_h = 8 if inrange(va133m1m,161,169)
replace racod_h = 9 if inrange(va133m1m,171,179)
replace racod_h = 10 if inrange(va133m1m,181,189)
replace racod_h = 11 if inrange(va133m1m,191,196)
replace racod_h = 12 if inrange(va133m1m,595,597)
replace racod_h = 13 if va133m1m==997

*** Drop Wave 9 Exit Data for Causes of Death
drop `coverscreen_w9_exit'

*********************************************************************

*** Set Wave Number
local wv=10
local pre_wv=9

*** Bring Wave 10 Exit Data for Causes of Death
local coverscreen_w10_exit wa028 wa131 wa133m1m wa133m2m wz145  
merge 1:1 hhidpn using "$wave_10_exit", keepusing(`coverscreen_w10_exit') nogen

*** In exit interview - Wave 10
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(wz145,.,2,3)
replace inw`wv'xt = 1 if wz145==. & inrange(wa028,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing wa131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = wa131 if inrange(wa131,1,2)
replace radexpec = 3 if wa131==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing wa133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if wa133m1m==998
replace ragcod = .r if wa133m1m==999
replace ragcod = .c if inrange(wa133m1m,601,607)
replace ragcod = .n if wa133m1m==996
replace ragcod = 1 if inrange(wa133m1m,101,102)
replace ragcod = 2 if inrange(wa133m1m,121,129)
replace ragcod = 3 if wa133m1m==103 | inrange(wa133m1m,111,119) | inrange(wa133m1m,131,597) | wa133m1m==997

*** What was the major illness that led to (his/her) death?
missing wa133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if wa133m1m==998
replace racod_h = .r if wa133m1m==999
replace racod_h = .c if inrange(wa133m1m,601,607)
replace racod_h = .n if wa133m1m==996
replace racod_h = 1 if inrange(wa133m1m,101,102)
replace racod_h = 2 if wa133m1m==103
replace racod_h = 3 if inrange(wa133m1m,111,119)
replace racod_h = 4 if inrange(wa133m1m,121,129)
replace racod_h = 5 if inrange(wa133m1m,131,139)
replace racod_h = 6 if inrange(wa133m1m,141,149)
replace racod_h = 7 if inrange(wa133m1m,151,159)
replace racod_h = 8 if inrange(wa133m1m,161,169)
replace racod_h = 9 if inrange(wa133m1m,171,179)
replace racod_h = 10 if inrange(wa133m1m,181,189)
replace racod_h = 11 if inrange(wa133m1m,191,196)
replace racod_h = 12 if inrange(wa133m1m,595,597)
replace racod_h = 13 if wa133m1m==997

*** Drop Wave 10 Exit Data for Causes of Death
drop `coverscreen_w10_exit'

*********************************************************************

*** Set Wave Number
local wv=11
local pre_wv=10

*** Bring Wave 11 Exit Data for Causes of Death
local coverscreen_w11_exit xa028 xa131 xa133m1m xa133m2m xz145 
merge 1:1 hhidpn using "$wave_11_exit", keepusing(`coverscreen_w11_exit') nogen

*** In exit interview - Wave 11
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(xz145,.,2,3)
replace inw`wv'xt = 1 if xz145==. & inrange(xa028,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing xa131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = xa131 if inrange(xa131,1,2)
replace radexpec = 3 if xa131==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing xa133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if xa133m1m==998
replace ragcod = .r if xa133m1m==999
replace ragcod = .c if inrange(xa133m1m,601,607)
replace ragcod = .n if xa133m1m==996
replace ragcod = 1 if inrange(xa133m1m,101,102)
replace ragcod = 2 if inrange(xa133m1m,121,129)
replace ragcod = 3 if xa133m1m==103 | inrange(xa133m1m,111,119) | inrange(xa133m1m,131,597) | xa133m1m==997

*** What was the major illness that led to (his/her) death?
missing xa133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if xa133m1m==998
replace racod_h = .r if xa133m1m==999
replace racod_h = .c if inrange(xa133m1m,601,607)
replace racod_h = .n if xa133m1m==996
replace racod_h = 1 if inrange(xa133m1m,101,102)
replace racod_h = 2 if xa133m1m==103
replace racod_h = 3 if inrange(xa133m1m,111,119)
replace racod_h = 4 if inrange(xa133m1m,121,129)
replace racod_h = 5 if inrange(xa133m1m,131,139)
replace racod_h = 6 if inrange(xa133m1m,141,149)
replace racod_h = 7 if inrange(xa133m1m,151,159)
replace racod_h = 8 if inrange(xa133m1m,161,169)
replace racod_h = 9 if inrange(xa133m1m,171,179)
replace racod_h = 10 if inrange(xa133m1m,181,189)
replace racod_h = 11 if inrange(xa133m1m,191,196)
replace racod_h = 12 if inrange(xa133m1m,595,597)
replace racod_h = 13 if xa133m1m==997

*** Drop Wave 11 Exit Data for Causes of Death
drop `coverscreen_w11_exit'

*********************************************************************

*** Set Wave Number
local wv=12
local pre_wv=11

*** Bring Wave 12 Exit Data for Causes of Death
local coverscreen_w12_exit ya028 ya131 ya133m1m ya133m2m yz145 
merge 1:1 hhidpn using "$wave_12_exit", keepusing(`coverscreen_w12_exit') nogen

*** In exit interview - Wave 12
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(yz145,.,2,3)
replace inw`wv'xt = 1 if yz145==. & inrange(ya028,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing ya131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = ya131 if inrange(ya131,1,2)
replace radexpec = 3 if ya131==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing ya133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if ya133m1m==998
replace ragcod = .r if ya133m1m==999
replace ragcod = .c if inrange(ya133m1m,601,607)
replace ragcod = .n if ya133m1m==996
replace ragcod = 1 if inrange(ya133m1m,101,102)
replace ragcod = 2 if inrange(ya133m1m,121,129)
replace ragcod = 3 if ya133m1m==103 | inrange(ya133m1m,111,119) | inrange(ya133m1m,131,597) | ya133m1m==997

*** What was the major illness that led to (his/her) death?
missing ya133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if ya133m1m==998
replace racod_h = .r if ya133m1m==999
replace racod_h = .c if inrange(ya133m1m,601,607)
replace racod_h = .n if ya133m1m==996
replace racod_h = 1 if inrange(ya133m1m,101,102)
replace racod_h = 2 if ya133m1m==103
replace racod_h = 3 if inrange(ya133m1m,111,119)
replace racod_h = 4 if inrange(ya133m1m,121,129)
replace racod_h = 5 if inrange(ya133m1m,131,139)
replace racod_h = 6 if inrange(ya133m1m,141,149)
replace racod_h = 7 if inrange(ya133m1m,151,159)
replace racod_h = 8 if inrange(ya133m1m,161,169)
replace racod_h = 9 if inrange(ya133m1m,171,179)
replace racod_h = 10 if inrange(ya133m1m,181,189)
replace racod_h = 11 if inrange(ya133m1m,191,196)
replace racod_h = 12 if inrange(ya133m1m,595,597)
replace racod_h = 13 if ya133m1m==997

*** Drop Wave 12 Exit Data for Causes of Death
drop `coverscreen_w12_exit'

*********************************************************************

*** Set Wave Number
local wv=13
local pre_wv=12

*** Bring Wave 13 Exit Data for Causes of Death
local coverscreen_w13_exit za028 za131 za133m1m za133m2m zz145 
merge 1:1 hhidpn using "$wave_13_exit", keepusing(`coverscreen_w13_exit') nogen

*** In exit interview - Wave 13
gen inw`wv'xt = .
replace inw`wv'xt = 0 if inlist(zz145,.,2,3,4)
replace inw`wv'xt = 1 if zz145==. & inrange(za028,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing za131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = za131 if inrange(za131,1,2)
replace radexpec = 3 if za131==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing za133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if za133m1m==998
replace ragcod = .r if za133m1m==999
replace ragcod = .c if inrange(za133m1m,601,607)
replace ragcod = .n if za133m1m==996
replace ragcod = 1 if inrange(za133m1m,101,102)
replace ragcod = 2 if inrange(za133m1m,121,129)
replace ragcod = 3 if za133m1m==103 | inrange(za133m1m,111,119) | inrange(za133m1m,131,597) | za133m1m==997

*** What was the major illness that led to (his/her) death?
missing za133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if za133m1m==998
replace racod_h = .r if za133m1m==999
replace racod_h = .c if inrange(za133m1m,601,607)
replace racod_h = .n if za133m1m==996
replace racod_h = 1 if inrange(za133m1m,101,102)
replace racod_h = 2 if za133m1m==103
replace racod_h = 3 if inrange(za133m1m,111,119)
replace racod_h = 4 if inrange(za133m1m,121,129)
replace racod_h = 5 if inrange(za133m1m,131,139)
replace racod_h = 6 if inrange(za133m1m,141,149)
replace racod_h = 7 if inrange(za133m1m,151,159)
replace racod_h = 8 if inrange(za133m1m,161,169)
replace racod_h = 9 if inrange(za133m1m,171,179)
replace racod_h = 10 if inrange(za133m1m,181,189)
replace racod_h = 11 if inrange(za133m1m,191,196)
replace racod_h = 12 if inrange(za133m1m,595,597)
replace racod_h = 13 if za133m1m==997

*** Drop Wave 13 Exit Data for Causes of Death
drop `coverscreen_w13_exit'

*********************************************************************

*** Set Wave Number
local wv=14
local pre_wv=13

*** Bring Wave 14 Exit Data for Causes of Death
local coverscreen_w14_exit xqa028 xqa131 xqa133m1m xqa133m2m xqz145 				   
merge 1:1 hhidpn using "$wave_14_exit", keepusing(`coverscreen_w14_exit') nogen

*** In exit interview - Wave 14
gen inw`wv'xt = 0 //.
replace inw`wv'xt = 0 if inlist(xqz145,.,2,3,4)
replace inw`wv'xt = 1 if xqz145==. & inrange(xqa028,1,5)
label variable inw`wv'xt "inw`wv'xt: r in w`wv' exit interview"
label value inw`wv'xt inwxt

*** Was the death expected at about the time it occurred, or was it unexpected?
missing xqa131 if inw`wv'xt==1, result(radexpec) wave(`wv')
replace radexpec = xqa131 if inrange(xqa131,1,2)
replace radexpec = 3 if xqa131==7

*** What was the major illness that led to (his/her) death?
* Grouped
missing xqa133m1m if inw`wv'xt==1, result(ragcod) wave(`wv')
replace ragcod = .d if xqa133m1m==998
replace ragcod = .r if xqa133m1m==999
replace ragcod = .c if inrange(xqa133m1m,601,607)
replace ragcod = .n if xqa133m1m==996
replace ragcod = 1 if inrange(xqa133m1m,101,102)
replace ragcod = 2 if inrange(xqa133m1m,121,129)
replace ragcod = 3 if xqa133m1m==103 | inrange(xqa133m1m,111,119) | inrange(xqa133m1m,131,597) | xqa133m1m==997

*** What was the major illness that led to (his/her) death?
missing xqa133m1m if inw`wv'xt==1, result(racod_h) wave(`wv')
replace racod_h = .d if xqa133m1m==998
replace racod_h = .r if xqa133m1m==999
replace racod_h = .c if inrange(xqa133m1m,601,607)
replace racod_h = .n if xqa133m1m==996
replace racod_h = 1 if inrange(xqa133m1m,101,102)
replace racod_h = 2 if xqa133m1m==103
replace racod_h = 3 if inrange(xqa133m1m,111,119)
replace racod_h = 4 if inrange(xqa133m1m,121,129)
replace racod_h = 5 if inrange(xqa133m1m,131,139)
replace racod_h = 6 if inrange(xqa133m1m,141,149)
replace racod_h = 7 if inrange(xqa133m1m,151,159)
replace racod_h = 8 if inrange(xqa133m1m,161,169)
replace racod_h = 9 if inrange(xqa133m1m,171,179)
replace racod_h = 10 if inrange(xqa133m1m,181,189)
replace racod_h = 11 if inrange(xqa133m1m,191,196)
replace racod_h = 12 if inrange(xqa133m1m,595,597)
replace racod_h = 13 if xqa133m1m==997

*** Drop Wave 14 Exit Data for Causes of Death
drop `coverscreen_w14_exit'

/********************************************************************
Prepare Main Dataset
********************************************************************/

*** In exit interview
gen inxt = .
replace inxt = 0 if inw2xt==0 | inw3xt==0 | inw4xt==0 | inw5xt==0 | inw6xt==0 | inw7xt==0 | ///
										inw8xt==0 | inw9xt==0 | inw10xt==0 | inw11xt==0 | inw12xt==0 | inw13xt==0 | inw14xt==0
replace inxt = 1 if inw2xt==1 | inw3xt==1 | inw4xt==1 | inw5xt==1 | inw6xt==1 | inw7xt==1 | ///
										inw8xt==1 | inw9xt==1 | inw10xt==1 | inw11xt==1 | inw12xt==1 | inw13xt==1 | inw14xt==1
label variable inxt "inxt: r in exit interview"
label values inxt inwxt

*** Drop respondents not in the RAND HRS
drop if in_rndhrs != 1
drop in_rndhrs

*** Drop respondents without Exit Data
drop if inw2xt!=1 & inw3xt!=1 & inw4xt!=1 & inw5xt!=1 & inw6xt!=1 & inw7xt!=1 & inw8xt!=1 & inw9xt!=1 & inw10xt!=1 & inw11xt!=1 & inw12xt!=1 & inw13xt!=1 & inw14xt!=1

*** Drop RAND variables
drop inw1 inw2 inw3 inw4 inw5 inw6 inw7 inw8 inw9 inw10 inw11 inw12 inw13 inw14 h3hhid h4hhid 

*** Update all value labels
foreach v of var * {
	local vlabel : value label `v'
	if "`vlabel'" != "" {
		label define `vlabel' ///
			.r ".r:refuse" ///
			.m ".m:missing" ///
			.d ".d:dk", modify
	}
}

*** Keep important variables
keep hhid pn hhidpn radexpec ragcod racod_h

*** Generate Simplified Cause of Death
gen rascod = racod_h
replace rascod = 15 if missing(racod_h)
replace rascod = 14 if racod_h==.c
label values rascod simplcauses

label variable rascod "rascod: disease that caused r's death (incl. not a health)"

/*
tab rascod, m

  rascod: disease that caused r's death |
                   (incl. not a health) |      Freq.     Percent        Cum.
----------------------------------------+-----------------------------------
                      1.cancers, tumors |      3,515       23.06       23.06
                      2.skin conditions |         24        0.16       23.22
3.musculoskeletal system, connective ti |        189        1.24       24.46
 4.heart, circulatory, blood conditions |      5,323       34.92       59.38
5.allergies, hayfever, sinusitis, tonsi |      1,693       11.11       70.49
6.endocrine, metabolic, nutritional con |        486        3.19       73.68
7.digestive system (stomach, liver, gal |      1,044        6.85       80.53
     8.neurological, sensory conditions |        339        2.22       82.75
9.reproductive system, prostate conditi |          7        0.05       82.80
 10.emotional, psychological conditions |         56        0.37       83.16
                       11.miscellaneous |        405        2.66       85.82
                      12.other symptoms |        650        4.26       90.09
              13.other health condition |        948        6.22       96.31
              14.not a health condition |        156        1.02       97.33
                             15.missing |        407        2.67      100.00
----------------------------------------+-----------------------------------
                                  Total |     15,242      100.00
*/

/********************************************************************
Save Main Dataset
********************************************************************/

*** Define variable order
order hhidpn ///
      hhid ///
      pn ///
      radexpec ///
      racod_h ///
      ragcod ///
	  rascod
									
*** Compress dataset
compress

***save output dataset
save "`output_path'/DeathCause.dta", replace
export delimited using "`output_path'/DeathCause.csv", replace
